/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/solution.c,v $
 * $Id: solution.c,v 3.17 1999/10/15 23:56:56 heiner Exp $
 *
 *	Print solution tree, depth limited.
 */

#include "types.h"
#include "board.h"
#include "job.h"
#include "analyse.h"
#include "mlsubr.h"
#include "move.h"
#include "move_gen.h"
#include <stdio.h>
#include "output.h"
#include "mgsubr.h"		/* mg_link */
#include "solopt.h"
#include "solution.h"


#ifndef SOL_USE_TABS
# define SOL_USE_TABS	1	/* CF: whether to use tab for indenting */
#endif

#ifndef SOL_SHOW_DEPTH
# define SOL_SHOW_DEPTH	1	/* CF: whether to print solution depth at ... */
#endif

Eximpl int	f_nodualdep =0;	/* no duals for depth <= this */

static int	id_curr;	/* current (logical) indent */
static int	id_want;	/* wanted (logical) indent */

#define CHAR_IN_MOVE	6	/* cf. put__move() */


    Eximpl char*			/* --> static data */
fill_conti( int soldep )
{
    static char	sym_cont[]	={ "..." };
#if SOL_SHOW_DEPTH
    static char	buf[50];

    if( soldep < 0 ) {		/* shall not be printed */
	return sym_cont;
    }
    soldep -= 1;		/* to be printed */
    if( soldep < 100 ) {
	sprintf(buf, "%s+%d", sym_cont  , soldep);
    }else {
	sprintf(buf, "%s+%d", sym_cont+1, soldep);
    }
    return buf;
#else	/* ! SOL_SHOW_DEPTH */
    return sym_cont;
#endif	/* ! SOL_SHOW_DEPTH */
}


/*
 * Synchronize wanted and current indent.
 */
    static void
id_sync(void)
{
    int	cols;

    if( id_curr > id_want ) {		/* space back to a new output line */
	printf("\n"); id_curr = 0;
    }
    if( id_curr < id_want ) {		/* space forward to wanted indent */
	cols = (id_want - id_curr) * (1 + CHAR_IN_MOVE);
#if SOL_USE_TABS			/* use tabulators */
	if( id_curr == 0 ) {
	    while( cols >= 8 ) {
		printf("\t"); cols -= 8;
	    }
	}
#endif
	if( cols > 0 ) {
	    printf("%*c", cols, ' ');
	}
	id_curr = id_want;
    }
}


    static void
app_solution( register Board* bp, const Move* mp )
{
    id_sync();
#if CHAR_IN_MOVE >= 9
    printf(" ");
    put__move(bp, mp);
#else
# if CHAR_IN_MOVE <= 6
    printf(" %-*.*s", CHAR_IN_MOVE, CHAR_IN_MOVE, mvc_L6(bp, mp));
# else
    printf(" %-*.*s", CHAR_IN_MOVE, CHAR_IN_MOVE, mvc_L7(bp, mp));
# endif
#endif
    ++id_curr;
}


    static void
app_conti( int soldep )
{
    id_sync();
    printf(" %s", fill_conti(soldep));
    ++id_curr;
}


    static void
sol_1move_list( Move* mp, Bool valid, Movelist* sols )
{
    clear_list(sols);
    if( mp && valid && !no_pos(mp->m_from) ) {
	app_move(mp, sols);
    }
    mg_link(sols);
}


    static void
sol_1redu_list( Movelist* sols )
{
    if( list_length(sols) > 1 ) {
	    Move	move;
	move = *(sols->l_first);
	sol_1move_list(&move, TRUE, sols);
    }
}


    Eximpl int				/* from analyse */
sol_analyse( Board* bp, int depth, Movelist* sols )
{
    /*
     * Here we know, how to call "analyse".
     * We also manage dual-suppression.
     */
    int		res;

    if( depth <= f_nodualdep ) {
	/*
	 * We ask analyse for just one move, and fill that one
	 * into the given list.
	 */
	    Move	move;
	move.m_from = NO_POS;		/* not yet filled */
	res = analyse(bp, depth, (Movelist*)0, &move, (RefuList*)0);
	sol_1move_list(&move, ANASUC(res), sols);
    }else {		/* normal, all solutions */
	res = analyse(bp, depth, sols, (Move*)0, (RefuList*)0);
	/*
	 * Now, if the resulting depth is smaller than the asked depth,
	 * we might fall below the dual suppression limit.
	 * In this case the list is to be truncated after the first move.
	 */
	if( (list_length(sols) > 1)		/* something to cut off */
	 && ANASUC(res) && (ANADEP(res) <= f_nodualdep) ) {
	    sol_1redu_list(sols);
	}
    }
    return res;
}

    static int			/* from analyse */
sol_ana_help( Board* bp, int plies, Movelist* sols )
{
    int		res;
    int		depth;

    depth = plies / 2;		/* FFS: check against chest.c */
    if( depth <= f_nodualdep ) {
	    Move	move;
	move.m_from = NO_POS;		/* not yet filled */
	res = ana_help(bp, plies, (Movelist*)0, &move);
	sol_1move_list(&move, ANASUC(res), sols);
    }else {
	res = ana_help(bp, plies, sols, (Move*)0);
	if( (list_length(sols) > 1)		/* something to cut off */
	 && ANASUC(res) && (ANADEP(res)/2 /*FFS*/ <= f_nodualdep) ) {
	    sol_1redu_list(sols);
	}
    }
    return res;
}


static void	pr_solution(Board* bp, int depth, Movelist* solp, int limdepth);

    static void
pr_defends(
    register Board*	bp,
    int			subdepth,
    Movelist*		defp,
    int			sublimdepth)
{
    register const Move*	dp;
    register int		res;
    Movelist			subsols;

    ml_ck_attr(bp, defp);
    formoves( defp, dp ) {
	app_solution(bp, dp);
	if( subdepth > 0 ) {
	    move_execute(bp, dp);
	    ++id_want;
	    res = sol_analyse(bp, subdepth, &subsols);
	    pr_solution(bp, ANADEP(res), &subsols, sublimdepth);
	    --id_want;
	    move_undo(bp);
	}
    }
}

    static void
pr_solution(
    register Board*		bp,
    int				depth,
    Movelist*			solp,
    int				limdepth)
{
    register const Move*	ap;
    Movelist			defends;

    ml_ck_attr(bp, solp);
    formoves( solp, ap ) {
	app_solution(bp, ap);
	if( (depth > 1) || (f_jobattr & JTA_2plies) ) {
	    move_execute(bp, ap);
	    ++id_want;
	    if( move_gen(bp, &defends) ) {	/* exist defender moves */
		if( limdepth <= 1 ) {
		    app_conti(depth);
		}else {
		    pr_defends(bp, depth-1, &defends, limdepth-1);
		}
	    }
	    --id_want;
	    move_undo(bp);
	}
    }
}


    static void
pr_help_solution(
    register Board*		bp,
    int				plies,
    Movelist*			solp,
    int				limplies)
{
    register const Move*	mp;
    register int		res;
    Movelist			subsols;

    ml_ck_attr(bp, solp);
    formoves( solp, mp ) {
	app_solution(bp, mp);
	++id_want;
	if( (plies > 1) && (limplies <= 1) ) {
	    app_conti(plies/2);		/* more to come, but cut off */
	}else {
	    move_execute(bp, mp);
	    if( plies > 1 ) {
		res = sol_ana_help(bp, plies-1, &subsols);
		pr_help_solution(bp, ANADEP(res), &subsols, limplies-1);
	    }
	    move_undo(bp);
	}
	--id_want;
    }
}


    Eximpl void
solution_stats( Bool doprint )
{
    solopt_stats(doprint);
}


/*
 * print_solution()
 *	The given list of moves is the top level of the solution
 *	for the specified board and the specified depth.
 *	Print the (complete) solution tree.
 */
    Eximpl void
print_solution(
    Board*	bp,			/* for this board */
    int		depth,			/* solutions in so many moves */
    int		preply,			/* + this prefix ply */
    Movelist*	solp,			/* are stored here */
    int		limdepth,		/* max depth to print */
    int		wantoptim)		/* whether optimization wanted */
{
#if 1
    IF_HEINER_SET(01,
	    char	buf[200];
	sol_pv_fill(bp, depth, preply, solp->l_first, buf, sizeof(buf));
	printf("PV=%s;\n", buf);
    )
#endif
    if( depth < 1 ) {		/* FFS already solved, nothing to print */
	return;
    }
    if( !(f_jobattr & JTA_help) && wantoptim ) {
	init_solopt();
	solopt_print(bp, depth, preply, solp, limdepth);
	return;
    }
    id_curr = 0;
    id_want = 0;
    if( f_jobattr & JTA_help ) {
	pr_help_solution(bp, preply + depth*2, solp, limdepth*2);
    }else {
	if( preply ) {
	    pr_defends(bp, depth, solp, limdepth);
	}else {
	    pr_solution(bp, depth, solp, limdepth);
	}
    }
    id_sync();
}

/*===========================================================================*/
/* Generation of PVs (pricipal variations):
 * - always only one move (defender & attacker).
 * - use SAN.
 */

    static int				/* so many chars appended */
sol_pv__1app(
    Board*	bp,			/* for this board */
    Move*	mp,			/* one move is stored here */
    char*	buf,			/* fill in, here */
    int		siz)			/* physical size of "buf" */
{
    const char*	cp;
    int		len;

    len = 0;
    if( mp && !no_pos(mp->m_from) ) {
	mv_ck_attr(bp, mp);		/*FFS*/
	cp  = mvc_SAN(bp, mp);
	len = 1 + str_len(cp);
	if( len < siz ) {
	    sprintf(buf, " %s", cp);	/* leading blank, terminating NUL */
	}else {
	    len = 0;			/* move does not fit: omitted */
	}
    }
    return len;
}

static void sol_pv_app(Board* bp, int depth, Move* solp, char* buf, int siz);

    static Bool			/* whether really done it */
sol_pv_def_app(
    Board*	bp,			/* for this board */
    int		subdepth,		/* and a subsolution in so many moves */
    Move*	defp,			/* the defender is stored here */
    char*	buf,			/* fill in, here */
    int		siz)			/* physical size of "buf" */
{
    Bool	defok;
    Move	subsol;
    int		res;
    int		len;

    /*
     * We must assure, that there is a subsolution of the specified depth,
     * and fail, if there is none following this move.
     * NB: analyse() does grok "helpmate in 0 moves".
     */
    defok = FALSE;
    subsol.m_from = NO_POS;
    if( (subdepth > 0) || (f_jobattr & JTA_help) ) {
	move_execute(bp, defp);
	res = analyse(bp, subdepth, (Movelist*)0, &subsol, (RefuList*)0);
	if( ANASUC(res) && (ANADEP(res) == subdepth) ) {
	    defok = TRUE;		/* take this maximal deep */
	}
	move_undo(bp);
    }else {
	if( (subdepth > 0) || (f_jobattr & JTA_2plies) ) {
	    /*
	     * For selfmate in 1, all "defenders" are part of
	     * the solution tree, so it is ok to take the first.
	     */
	    defok = TRUE;		/* just take the first */
	}
    }
    if( defok ) {
	len = sol_pv__1app(bp, defp, buf, siz);
	if( (len > 0) && (subdepth > 0) && !no_pos(subsol.m_from) ) {
	    move_execute(bp, defp);
	    sol_pv_app(bp, subdepth, &subsol, buf+len, siz-len);
	    move_undo(bp);
	}
    }
    return defok;
}

    static void
sol_pv_app(
    Board*	bp,			/* for this board */
    int		depth,			/* a solution in so many moves */
    Move*	solp,			/* is stored here */
    char*	buf,			/* fill in, here */
    int		siz)			/* physical size of "buf" */
{
    int		len;

    if( siz < (2+1) ) {		/* no move can fit in, there */
	return;
    }
    len = sol_pv__1app(bp, solp, buf, siz);
    if( len <= 0 ) {
	return;
    }
    buf += len;
    siz -= len;
    if( (depth > 1) || (f_jobattr & JTA_2plies) ) {
	    Movelist	defends;
	    Move*	dp;
	move_execute(bp, solp);
	if( move_gen(bp, &defends) ) {		/* exist defender moves */
	    ml_ck_attr(bp, &defends);
	    formoves( &defends, dp ) {
		/*
		 * NB: analyse() does grok "helpmate in 0 moves".
		 */
		if( sol_pv_def_app(bp, depth-1, dp, buf, siz) ) {
		    break;			/* for defends */
		}
	    }
	}
	move_undo(bp);
    }
}


    Eximpl void
sol_pv_fill(
    Board*	bp,			/* for this board */
    int		depth,			/* a solution in so many moves */
    int		preply,			/* + this prefix ply */
    Move*	solp,			/* is stored here */
    char*	buf,			/* fill in, here */
    int		siz)			/* physical size of "buf" */
{
#if 0
    printf("PVfill: bp=%lx dep=%d+%d solp=%lx buf=%lx[%d]\n",
	    (long)bp, depth, preply, (long)solp, (long)buf, siz);
    exit(0);
#endif
    if( siz > 0 ) {
	buf[0] = 0;			/* assure proper termination */
    }
    if( solp ) {
	    int		fsave;		/* FFS: here or at caller? */
	fsave = f_mvtrace;
	f_mvtrace = 0;
	if( preply ) {
	    mv_ck_attr(bp, solp);	/* FFS: necessary? */
	    (void) sol_pv_def_app(bp, depth, solp, buf, siz);
	}else {
	    if( depth > 0 ) {		/* otherwise solp not filled */
		mv_ck_attr(bp, solp);	/* FFS: necessary? */
		sol_pv_app(bp, depth, solp, buf, siz);
	    }
	}
	f_mvtrace = fsave;
    }
}
