/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/timing.c,v $
 * $Id: timing.c,v 3.8 1999/12/04 23:07:17 heiner Exp $
 *
 *	timing assistance
 *
 * Standard C offers:
 * - time()	is wall clock time and has resolution 1 second.
 *		Differences must be computed via difftime().
 * - clock()	is effective processor time (we want this).
 *		Its resolution: CLOCKS_PER_SEC is typically 1000000,
 *		but clock_t is typically 32 bit signed, only:
 *		2^31*10^-6sec <= 2147 sec <= 36 min
 *		Also, we do not know its behaviour when overflowing.
 *
 * BSD-style UNIX typically offers:
 * - getrusage()
 *		(struct rusage) contains the data we want.
 *		Resolution is seconds and micro seconds separated.
 *
 * SYSV-style UNIX typically offers:
 * - times()	(struct tms) contains the data we want.
 *		Resolution is given by sysconf(_SC_CLK_TCK),
 *		or the HZ manifest
 *
 * FFS: When we are neither BSD nor SYSV, we use time(), which is
 *	wall clock time.  Sorry.
 *	We could improve by using clock(), but we would need to be
 *	called often enough, in order to not miss any wrap-around.  FFS.
 */

#include "types.h"
#include "timing.h"


#if (!!TIM_USE_TIMES + !!TIM_USE_TIME + !!TIM_USE_RUSAGE + !!TIM_USE_STD) == 0
# if __STDC__
#  define TIM_USE_STD		1	/* portable, ANSI standard */
# else
#  if ! BSD
#   if ! defined(_WIN32)
#    define TIM_USE_TIMES	1	/* SYS-V standard */
#   else /* _WIN32 */
#    define TIM_USE_TIME	1	/* non-UNIX fall back */
#   endif
#  else /* BSD */
#   define TIM_USE_RUSAGE	1	/* BSD standard */
#  endif /* BSD */
# endif /* __STDC__ */
#endif

#if (!!TIM_USE_TIMES + !!TIM_USE_TIME + !!TIM_USE_RUSAGE + !!TIM_USE_STD) != 1
# include "/// >>> TIM_USE_ misconfigured <<< ///"
#endif

#if TIM_USE_TIMES
# include <sys/times.h>
# include <unistd.h>
  typedef struct tms	TimeType;
# if 1				/* using sysconf */
   static long		hz	= 0;
# else				/* old version: use HZ */
#  include <sys/param.h>
   static long		hz	= HZ;
# endif
#endif

#if TIM_USE_RUSAGE
# include <sys/time.h>
# include <sys/resource.h>
  typedef struct rusage	TimeType;
#endif	/* TIM_USE_RUSAGE */

#if TIM_USE_TIME
# include <time.h>
# if defined(__STDC__)
#  define diff_time(t1,t0)	difftime(t1,t0)
# else
#  define diff_time(t1,t0)	((double)(t1)-(t0))
# endif
  typedef time_t	TimeType;
  static TimeType	g_bas;		/* need a basis for timing */
  static int		g_bas_ok = 0;	/* whether g_bas is filled */
#endif	/* TIM_USE_TIME */

#if TIM_USE_STD
# include <time.h>
# if defined(__STDC__)
#  define diff_time(t1,t0)	difftime(t1,t0)
# else
#  define diff_time(t1,t0)	((double)(t1)-(t0))
# endif
  typedef struct TimeType
  {
    time_t	real;
    clock_t	virt;
    int		filled;			/* 01=real, 02=virt */
  }			TimeType;
  static TimeType	g_bas;		/* need a basis for timing */
  static int		g_bas_ok = 0;	/* whether g_bas is filled */
  static TimeSecs	g_sum    = 0;	/* collected before g_bas */
  static int		g_mix    = 0;	/* filled types (real/virt) */
#endif	/* TIM_USE_STD */

static TimeType		g_cur;		/* fill all info here by sys-call */


/*
 * get_self()
 *	Fill the global usage object "g_cur".
 */
    static int			/* success */
get_self(void)
{
#if TIM_USE_TIMES
    if( times(&g_cur) != (clock_t)-1 ) return 1;		/* ok */
#endif
#if TIM_USE_RUSAGE
    if( getrusage(RUSAGE_SELF, &g_cur) >= 0 ) return 1;		/* ok */
#endif
#if TIM_USE_TIME
    if( time(&g_cur) != (time_t)-1 ) return 1;			/* ok */
#endif
#if TIM_USE_STD
    g_cur.filled = 0;
    if( time(&g_cur.real)      != (time_t )-1 ) g_cur.filled |= 01;
    if( (g_cur.virt = clock()) != (clock_t)-1 ) g_cur.filled |= 02;
    if( g_cur.filled ) return 1;		/* something in there */
#endif	/* TIM_USE_STD */
    return 0;			/* sorry */
}


    Eximpl void
timing_init(void)
{
#if TIM_USE_TIME || TIM_USE_STD
    (void) get_self();
    g_bas    = g_cur;
    g_bas_ok = 1;
#endif	/* TIM_USE_TIME */
#if TIM_USE_STD
    g_sum    = 0;
    g_mix    = 0;
#endif
}


/*
 * secs_user()
 *	Return how many seconds we have spent up to now.
 *	This should include user time, but not system time.  FFS.
 */
    Eximpl TimeSecs
secs_user(void)
{
#if TIM_USE_TIME || TIM_USE_STD
    if( ! g_bas_ok ) timing_init();
#endif
    if( get_self() ) {
#if TIM_USE_TIMES
	if( ! hz ) {
	    hz = sysconf(_SC_CLK_TCK);
	    if( hz <= 0 ) {
		hz = 100;		/* guessing, FFS */
	    }
	}
	return (double)g_cur.tms_utime / (double)hz;
	/* FFS: + system time? */
#endif
#if TIM_USE_RUSAGE
	return ((double)g_cur.ru_utime.tv_usec) / (double)1000000
	    +   (double)g_cur.ru_utime.tv_sec;
	/* FFS: + system time? */
#endif
#if TIM_USE_TIME
	return diff_time(g_cur, g_bas);
#endif
#if TIM_USE_STD
	if( (g_cur.filled & 02) && (g_cur.virt >= g_bas.virt) ) {
	    g_sum += (double)(g_cur.virt - g_bas.virt) / CLOCKS_PER_SEC;
	    g_bas  = g_cur;
	    g_mix |= 02;
	}else if( g_cur.filled & 01 ) {
	    g_mix |= 01;
	    g_sum += diff_time(g_cur.real, g_bas.real);
	    g_bas  = g_cur;
	}
	return g_sum;
#endif	/* TIM_USE_STD */
    }
    return 0;			/* sorry, no info available */
}


    Eximpl const char*		/* descriptive text */
timing_type(void)
{
#if TIM_USE_TIMES
    return "user";
#endif
#if TIM_USE_RUSAGE
    return "user";
#endif
#if TIM_USE_TIME
    return "real";
#endif
#if TIM_USE_STD
    switch( g_mix & 03 ) {
     case 00:	return "none";
     case 01:	return "real";
     case 02:	return "virt";
     case 03:	break;
    }
    return             "mixed";
#endif
}


    Eximpl int			/* printf precision for time in seconds */
timing_prec(void)
{
#if TIM_USE_TIMES
    if( hz >= 500 ) return 3;
    if( hz >=  50 ) return 2;
    if( hz >=   5 ) return 1;
    return 0;
#endif
#if TIM_USE_RUSAGE
    return 2;			/* guess-work */
#endif
#if TIM_USE_TIME
    return 0;
#endif
#if TIM_USE_STD
    if( (g_mix & 03) == 02 ) {	/* virtual, only */
	if( (((clock_t)1) / 4) == 0 ) {		/* clock_t is integer */
	    if( CLOCKS_PER_SEC >= 500 ) return 3;
	    if( CLOCKS_PER_SEC >=  50 ) return 2;
	    if( CLOCKS_PER_SEC >=   5 ) return 1;
	}
    }
    return 0;
#endif
}

/*=============================================================================
 * Portable Section: implementing one global timer
 */

typedef struct Timer	Timer;

struct Timer
{
    TimeSecs	sum;
    TimeSecs	beg;
};

static Timer	g_timer	={ 0, 0 };

    Eximpl void
gtimer_start(void)
{
    g_timer.beg = secs_user();
}

    Eximpl TimeSecs
gtimer_now(void)
{
    TimeSecs	now;

    now = secs_user();
    return g_timer.sum + (now - g_timer.beg);
}
