/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/trace.c,v $
 * $Id: trace.c,v 3.11 1999/10/20 23:59:03 heiner Exp $
 *
 *	tracing and interface to supervisor process
 *
 *	WARNING: when changing in the bodies of the trc_*() function,
 *		 either use TRC_GENUINE, or update the TRC_*() macros
 *		 in "trace.h".
 */

#include "types.h"
#include "board.h"
#include "output.h"
#include "cost.h"
#include "move.h"
#include <stdio.h>
#include "stats.h"
#include "trace.h"


#if HAS_INTERFACE
# include <errno.h>
# if IRIX
#  define _BSD_SIGNALS
# endif	/* IRIX */
# include <signal.h>
# include "interface.h"
# include "sysdep.h"
# include "bsd.h"
# ifndef SIGHND_RET_TYPE
#  define SIGHND_RET_TYPE	void
#  define SIGHND_RETURN(x)	/*empty*/
# endif
# if USG || defined(__STDC__)
#  include <unistd.h>		/* for read, write */
# endif
#endif

#if HAS_INTERFACE
Eximpl int	f_ic	  = 0;	/* whether we talk to an interactive chest */
Eximpl int	f_ic_skip = 0;	/* interactive chest */
Eximpl int	f_no_trc  = 0;
static int	tmp_no_mvtrc	= 0;
#else	/* ! HAS_INTERFACE */
# define	tmp_no_mvtrc	0
#endif	/* ! HAS_INTERFACE */



/*
 * Trace execution and undoing of moves.
 */

#define MX_TRLEVS	64

static int	mvtr_level	= 0;
static uint8	lev_cnts[ MX_TRLEVS + 1 ];

#define LEV_HAS_ARR()	( ((unsigned)mvtr_level) < (unsigned)MX_TRLEVS )

#define INC_INDENT	5


/*
 * Global variables used in this module:
 *	f_mvtrace
 *		The current level for move tracing (option -[tT])
 *		on standard output.  How many halfmoves deep.
 *	mvtr_level
 *		How deep we currenly are with move tracing,
 *		independant of whether we currently print anything.
 *		Changed by "move_{,un}trace()".
 *	lev_cnts[]
 *		How many moves are (currently) executed on the levels
 *		up to the current.
 *	f_ic	Whether we talk to an interactive chest interface.
 *		Set once in initialization.
 *	ic_mvtr_level
 *	f_no_trc
 *		Whether (temporarily) move tracing towards the interface
 *		is to be suppressed.  This is set and reset in the
 *		answer heuristic.
 *	tmp_no_mvtrc
 *		Whether we internally decided that temporarily move
 *		tracing to the interface is to be suppressed.
 *		This is done between "trc_def_start()" and "trc_def_moves()",
 *		as any moves done there miss the context of a move list,
 *		and seem to not belong to the recursive analysis.
 */

#if 0		/* expanded / replaced by trc_restart */
/*
 * move_trace_samelevel()
 *	Inform this module that this level of moves is started over.
 *	Thus, when tracing it, the counts should restart from 0.
 */
    static void
move_trace_samelevel( int depth )
{
    if( f_mvtrace ) {
	if( LEV_HAS_ARR() ) lev_cnts[mvtr_level] = 0;
	if( mvtr_level == 0 ) {
	    printf("+++ depth %d ...\n", depth);
	    oflush();
	}
    }
}
#endif


/*
 * move_trace()
 *	The specified move is going to be executed.
 *	It might be wanted to appear in a trace output.
 */
static AnaCost	g_last_cost = 0;

    static void
move_trace( const Board* bp, const Move* mp )
{
    int			i;

    if( mvtr_level < f_mvtrace ) {		/* this one is to be traced */
	printf("+%10.0f ", (double)(apx_cost - g_last_cost));
	g_last_cost = apx_cost;
	if( LEV_HAS_ARR() ) {
	    lev_cnts[mvtr_level  ] += 1;	/* increment this level */
	    lev_cnts[mvtr_level+1]  = 0;	/* restart next level */
	}
	for( i=0 ; i<=mvtr_level ; ++i ) {
	    if( i < MX_TRLEVS ) {
		printf("%*d ", INC_INDENT-1, lev_cnts[i]);
	    }else {
		printf("%-*c", INC_INDENT, ',');
	    }
	}
	put_move(bp, mp);
	oflush();
    }
    ++mvtr_level;
}


    static void
move_untrace(void)
{
    --mvtr_level;
}


#if HAS_INTERFACE

static int	ic_mvtr_level	= 0;
static int	ic_max_level	= 64;
static int	ic_skip_lev     = 999;


    static int
my_send( const char* msg, int len )
{
    int	res;
    int	mask;

    if( len > MAX_MSG_SIZ ) {
	fprintf(stderr, "ILLEGAL LENGTH %d\n", len);
	sys_exit(1);
    }
    mask = sigblock(sigmask(SIGUSR1));
#if 0
    if ((res = send(MSG_DESC, msg, len, 0)) < 0)
#else
    if ((res = write(MSG_DESC, msg, len)) < 0)
#endif
    {
	perror("send");
	sys_exit(1);
    }
    (void) sigsetmask(mask);
    return res;
}


    static void
board_to64( register const Board* bp, register Board64*	bp64 )
{
    register int	i;
    register int	j;

    for( i=0 ; i<8 ; ++i ) {
	for( j=0 ; j<8 ; ++j ) {
	    bp64->b64_f[MK_POS64(i, j)] = bp->b_f[MK_POS(i, j)];
	}
    }

    bp64->b64_tomove    = bp->b_tomove;
    bp64->b64_ep        = bp->b_ep;

#define Copy(s,sk,t,tk)	bcopy((s)->sk, (t)->tk, sizeof((t)->tk))
    Copy(bp, b_castle   , bp64, b64_castle   );
    Copy(bp, b_piece    , bp64, b64_piece    );
    Copy(bp, b_max_piece, bp64, b64_max_piece);
    Copy(bp, b_cur_piece, bp64, b64_cur_piece);
    Copy(bp, b_fig_cnt  , bp64, b64_fig_cnt  );
    Copy(bp, b_fig_set  , bp64, b64_fig_set  );
#undef Copy
}


/*
 * ic_samelevel()
 *	Inform this module that this level of moves is started over.
 */
    static void
ic_samelevel( int depth )
{
    cm_mv_dep	msg;

    if( f_no_trc ) return;

    if( (ic_mvtr_level >= 0) && (ic_mvtr_level < ic_max_level) ) {
	if( ic_mvtr_level == 0 ) {
	    msg.type  = CM_MOVE_DEPTH;
	    msg.len   = sizeof(msg);
	    msg.level = ic_mvtr_level;
	    msg.depth = depth;
	    if( my_send((char*)&msg, sizeof(msg)) < 0 ) {
		perror("send");
		sys_exit(1);
	    }
	    ic_skip_lev = 999;
	    f_ic_skip   = 0;
	}else {
	    msg.type  = CM_CUR_MOVE_DEPTH;
	    msg.len   = sizeof(msg);
	    msg.depth = depth;
	    msg.level = ic_mvtr_level;
	    if( my_send((char*)&msg, sizeof(msg)) < 0 ) {
		perror("send");
		sys_exit(1);
	    }
	}
    }
}


/*
 * ic_move_trace()
 *	The specified move is going to be executed.
 */
    static void
ic_move_trace( const Board* bp, const Move* mp )
{
    if( f_no_trc ) return;

    if( ic_mvtr_level >= ic_skip_lev ) {
	++ic_mvtr_level;
	f_ic_skip = 1;
	return;
    }
    ic_skip_lev = 999;
    f_ic_skip   = 0;
    if( ic_mvtr_level < ic_max_level ) {	/* this one is to be traced */
	cm_mv	msg;
	
	msg.type  = CM_MOVE;
	msg.level = ic_mvtr_level;
	msg.len   = sizeof(msg);

	board_to64(bp, &msg.b);
	msg.m = *mp;

	if( my_send((char*)&msg, sizeof(msg)) < 0 ) {
	    perror("send");
	    sys_exit(1);
	}
    }
    ++ic_mvtr_level;
}


    static void
ic_move_untrace(void)
{
    if( f_no_trc ) return;
    --ic_mvtr_level;
}


    static void
ic_trace_move_list( register const Board* bp, register const Movelist* mp )
{
    cm_mv_list	msg;
    int		nelem;

    if( f_no_trc ) return;

    if (ic_mvtr_level >= ic_skip_lev) {
	return;
    } 
    if( (ic_mvtr_level >= 0) && (ic_mvtr_level < ic_max_level) ) {
	msg.type   = CM_MOVE_LIST;

	board_to64(bp, &msg.b);

	nelem = list_length(mp);
	msg.m_list.l_attr  = mp->l_attr;
	msg.m_list.l_first = (Move *) (mp->l_first - mp->l_m);
	msg.m_list.l_free = (Move *) nelem;
	msg.level = ic_mvtr_level;
	nelem *= sizeof(Move);

	bcopy (mp->l_m, msg.m_list.l_m, nelem);
	msg.m_list_base = mp;
	msg.len    = sizeof(msg) - (MAX_MOVES * sizeof(Move) - nelem);
	if( my_send((char*)&msg, msg.len) < 0 ) {
	    perror("send");
	    sys_exit(1);
	}
    }
}
#endif	/* HAS_INTERFACE */


/*
 * Interface functions called from outside ...
 */


    Eximpl void		/*ARGSUSED*/
trc_into_analyse( const Board* bp, int depth )
{
}


    Eximpl void
trc_restart(void)
{
    /*
     * The current trace level wants its count to restart from 0.
     */
    if( f_mvtrace ) {
	if( LEV_HAS_ARR() ) lev_cnts[mvtr_level] = 0;
    }
    /* FFS: ic_samelevel( depth ) */
}


    Eximpl void
trc_att_start( int depth )
{
    if( f_mvtrace ) {
	if( LEV_HAS_ARR() ) lev_cnts[mvtr_level] = 0;	/* implicit restart */
	if( ((unsigned)mvtr_level) <= (unsigned)1 ) {
	    /* FFS: check also (mvtr_level < f_mvtrace) ? */
	    printf("+++ depth %d ...\n", depth);
	    oflush();
	}
    }
#if HAS_INTERFACE
    if( f_ic ) {
	ic_samelevel( depth );
    }
#endif
}


    Eximpl void
trc_att_end(void)
{
}


    Eximpl void
trc_def_start(void)
{
#if HAS_INTERFACE
    if( f_no_trc ) return;

    tmp_no_mvtrc = 1;
#endif
}

    Eximpl void
trc_def_end(void)
{
#if HAS_INTERFACE
    if( f_no_trc ) return;

    tmp_no_mvtrc = 0;
#endif
}


    Eximpl void
trc_att_moves( const Board* bp, const Movelist* lp )
{
#if HAS_INTERFACE
    if( f_ic ) {
	ic_trace_move_list( bp, lp );
    }
#endif
}


    Eximpl void
trc_def_moves( const Board* bp, const Movelist* lp )
{
#if HAS_INTERFACE
    if( f_no_trc ) return;

    tmp_no_mvtrc = 0;
    if( f_ic ) {
	ic_trace_move_list( bp, lp );
    }
#endif
}

    Eximpl void		/*ARGSUSED*/
trc_outof_analyse( const Board* bp, int retval, const Movelist* resp )
{
}


    Eximpl void
trc_move_exec( const Board* bp, const Move* mp )
{
#if 0
    fprintf(stderr, "%2d ic%d skip%d tnt%d fnt%d ilv%d skl%d\n",
		sc_move_exec % 100,	/*FFS*/
		f_ic, f_ic_skip,
		tmp_no_mvtrc, f_no_trc,
		ic_mvtr_level,
		ic_skip_lev
    );
    (void) fflush(stderr);
#endif
    if( f_mvtrace ) {
	move_trace(bp, mp);
    }
#if HAS_INTERFACE
    if( f_ic ) {
	if( !tmp_no_mvtrc ) {
	    ic_move_trace(bp, mp);
	}
    }
#endif
}

    Eximpl void
trc_move_undo(void)
{
    if( f_mvtrace ) {
	move_untrace();
    }
#if HAS_INTERFACE
    if( f_ic ) {
	if( !tmp_no_mvtrc ) {
	    ic_move_untrace();
	}
    }
#endif
}

    Eximpl void
trc_move_flush(void)
{
    /*
     * If currently a move would be traced, just add the final cost delta.
     */
    if( f_mvtrace ) {
	if( mvtr_level < f_mvtrace ) {
	    /*
	     * FFS: when there were no last move traced
	     */
	    printf("+%10.0f\n", (double)(apx_cost - g_last_cost));
	    g_last_cost = apx_cost;
	    oflush();
	}
    }
}


#if HAS_INTERFACE

/*
 * user_sig()
 *	Signal handler for SIGUSR1.
 *	We get it if there is data to read for us.
 */
    static SIGHND_RET_TYPE
user_sig( int signo )
{
    cm_chest_msg	m;
    int			res;

    signal(signo, user_sig);		/* re-install */
    if( (res = read(MSG_DESC, &m, sizeof(m))) != sizeof(m) ) {
	perror("read");
	fprintf(stderr, "READ: %d\n", res);
	sys_exit(1);
    }
    switch( m.type ) {
     default:
	fprintf(stderr, "illegal message type %ld\n", (long)m.type);
	sys_exit(1);
     case CM_SKIP:
	ic_skip_lev = m.val;
	break;
    }
    SIGHND_RETURN(0);
}

#endif	/* HAS_INTERFACE */


/*
 * trc_init()
 *	Initialize this module.
 *	When we are a subprocess of an interactive interface,
 *	we will detect that by a funny "argv[0]".
 */
    Eximpl void			/*ARGSUSED*/
trc_init( const char* argv0 )
{
#if HAS_INTERFACE
    if( !strcmp("//chest//", argv0) ) {
	f_ic = 1;
	signal(SIGUSR1, user_sig) ;
    }
#endif
}
